<?php
/**
 * Aptana Cloud Sample Application - Image Manipulation Demo Functions
 * 
 * This file contains all the function definitions for the sample application.
 * 
 * @author Ian Selby
 * @version 1.0
 * @license http://www.opensource.org/licenses/mit-license.php MIT License
 * @package ImageManipulationDemo
 * @filesource
 */

/**
 * The document site root
 */
define('SITE_ROOT', dirname(__FILE__) . '/');

/**
 * Valid file extensions for uploaded images
 *
 * @global array $GLOBALS['valid_files']
 * @name $valid_files
 */
$GLOBALS['valid_files'] = array('jpg', 'jpeg', 'gif', 'png');


/**
 * Uploads an Image to the Uploads Folder
 *
 * This function is pretty straight-forward.  It does a few simple things:
 *
 * - Attempts to chmod the uploads directory to make sure it is writeable
 * - Sanitizes the uploaded file name
 * - Uploads the file
 * - Makes sure that the file is a valid image type
 *
 * It will return an array with the following fields:
 *
 * - filename: the sanitized name of the uploaded file
 * - success: true / false
 * - error: the reason the upload failed (if success == false)
 * 
 * @param string $key The form uploader element to use for the upload
 * @return array The result of the upload
 */
function uploadImage($key)
{
	$return_array = array();

	// make sure the uploads directory exists, and create it if it doesn't
	if(!file_exists(SITE_ROOT . 'uploads'))
	{
		@mkdir(SITE_ROOT . 'uploads');
	}

	// make sure the uploads directory is writeable
	@chmod(SITE_ROOT . 'uploads', 0777);
	
	// clean up the filename
	$original_file = basename($_FILES[$key]['name']);
	$new_file = preg_replace('/[^a-zA-Z0-9._-]/', '', $original_file);
	// where we'll put the new file
	$target = SITE_ROOT . 'uploads/' . $new_file;
	
	// try to upload the file
	if(!move_uploaded_file($_FILES[$key]['tmp_name'], $target))
	{
		$return_array['success'] = false;
		$return_array['filename'] = $new_file;
		$return_array['error'] = 'Could not upload file';
		
		return $return_array;
	}
	
	// assume everything is ok
	$return_array['success'] = true;
	$return_array['filename'] = $new_file;
	$return_array['error'] = '';
	
	// get the extension, and make sure it's valid
	$extension = getFileExtension($target);
	
	// granted, it would be wiser to check the actual file header to verify it's an image,
	// but this is a sample so this should suffice :)
	if(!in_array($extension, $GLOBALS['valid_files']))
	{
		$return_array['success'] = false;
		$return_array['error'] = 'The uploaded file type is not valid';
	}
		
	return $return_array;
}

/**
 * Generates a "Polaroid" of an Image
 *
 * This function will take an image file, scale it down to no wider than 
 * 450 pixels, rotate it randomly, and then apply a "polaroid" type effect
 * to it with a drop-shadow.  It then returns the manipulated file name.
 *
 * Because of the effects applied during the manipulation, the returned file is
 * always a PNG image (for transparency support).
 *
 * @param string $filename Full filesystem path to the image to manipulate
 * @return string The name of the resultant file
 */
function generatePolaroidImage($filename)
{
	// build the new filename
	$extension = getFileExtension($filename);
	$new_file = str_ireplace($extension, 'png', $filename);

	// do all the magic ;)
	$bg = new ImagickDraw();
	$image = new Imagick($filename);
	$image->setImageFormat('png');
	$image->thumbnailImage(450, null);
	$image->setImageBackgroundColor(new ImagickPixel('black'));
    
    // come up with a random angle between 1 and 45 degrees
	$angle = mt_rand( 1, 45 );
	if ( mt_rand( 1, 2 ) % 2 === 0 ) 
	{   
		$angle = $angle * -1; 
	}   

	$image->polaroidImage($bg, $angle);
	$image->writeImage($new_file);
	$image->destroy();
	
	return basename($new_file);
}

/**
 * Returns the Extension of a File
 *
 * @param string $filename The full path to the file to get the extension of
 * @return string The extension of the file
 */
function getFileExtension($filename)
{
	$path_info = pathinfo($filename);
	return strtolower($path_info['extension']);
}

?>